<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\NotificationService;
use Illuminate\Http\Request;

class ActivationController extends Controller
{
    public function activate(\App\Http\Requests\ActivationRequest $request)
    {
        $code = \App\Models\ActivationCode::where('code', $request->code)->first();

        if ($code->status === 'expired' || ($code->expires_at && $code->expires_at->isPast())) {
            $code->update(['status' => 'expired']);
            return response()->json(['message' => 'الكود منتهي الصلاحية'], 403);
        }

        if ($code->status === 'inactive') {
            $code->update([
                'status'     => 'active',
                'expires_at' => \Carbon\Carbon::now()->addYear(),
                'user_id'    => $request->user()?->id, // Link to authenticated user if present
            ]);
        }

        // ── Fire Notifications ──────────────────────────────────────────────
        $userId   = $request->user()?->id ?? $code->user_id;
        $userName = $request->user()?->name ?? 'مستخدم غير معروف';

        // Admin notification
        NotificationService::admin(
            title:   'تفعيل كود جديد',
            message: "قام {$userName} بتفعيل الكود: {$code->code}",
            type:    'success',
            icon:    '🔑',
            meta:    [
                'code'      => $code->code,
                'user_id'   => $userId,
                'user_name' => $userName,
                'type'      => $code->type,
                'expires_at' => $code->expires_at?->toDateTimeString(),
            ]
        );

        // User notification (if we know who they are)
        if ($userId) {
            NotificationService::user(
                userId:  $userId,
                title:   'تم تفعيل اشتراكك بنجاح 🎉',
                message: "تم تفعيل الكود {$code->code} بنجاح. اشتراكك سيظل نشطاً حتى " . $code->expires_at?->translatedFormat('d F Y'),
                type:    'success',
                icon:    '✅',
                meta:    [
                    'code'       => $code->code,
                    'type'       => $code->type,
                    'expires_at' => $code->expires_at?->toDateTimeString(),
                ]
            );
        }

        // Multi-server URL logic (load balancing)
        $servers = [
            'https://server-eu-1.cdn.com/live/',
            'https://server-me-1.cdn.com/live/',
        ];
        $serverUrl = $servers[array_rand($servers)];

        return response()->json([
            'message'    => 'تم التفعيل بنجاح',
            'server_url' => $serverUrl,
            'token'      => $request->user()?->createToken('iptv-app')?->plainTextToken,
            'expiry'     => $code->expires_at?->toDateTimeString(),
        ]);
    }
}
