<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ActivationCode;
use App\Models\Channel;
use Illuminate\Http\Request;

class StreamController extends Controller
{
    /**
     * Return M3U playlist for authenticated user.
     * Called by the iptv-player: GET /api/user/playlist.m3u
     * Requires: Bearer token (Sanctum)
     */
    public function playlist(Request $request)
    {
        $user = $request->user();

        // Check active subscription
        $code = ActivationCode::where('user_id', $user->id)
            ->where('status', 'active')
            ->latest()
            ->first();

        if (!$code || ($code->expires_at && $code->expires_at->isPast())) {
            return response('# No active subscription', 403, ['Content-Type' => 'audio/mpegurl']);
        }

        $channels = Channel::where('status', 'online')->get();

        $m3u = "#EXTM3U\n";
        foreach ($channels as $channel) {
            $streamUrl = $channel->current_line === 'B' && $channel->line_b_url
                ? $channel->line_b_url
                : $channel->line_a_url;

            $group = $channel->category ?? 'عام';
            $m3u .= "#EXTINF:-1 tvg-name=\"{$channel->name}\" group-title=\"{$group}\",{$channel->name}\n";
            $m3u .= "{$streamUrl}\n";
        }

        return response($m3u, 200, [
            'Content-Type'        => 'audio/mpegurl',
            'Content-Disposition' => 'inline; filename="playlist.m3u"',
            'Access-Control-Allow-Origin' => '*',
        ]);
    }

    /**
     * Return stream info for the user (for the Watch button).
     * GET /api/user/stream-info
     */
    public function streamInfo(Request $request)
    {
        $user = $request->user();

        $code = ActivationCode::where('user_id', $user->id)
            ->where('status', 'active')
            ->latest()
            ->first();

        if (!$code || ($code->expires_at && $code->expires_at->isPast())) {
            return response()->json(['can_watch' => false, 'message' => 'لا يوجد اشتراك نشط'], 403);
        }

        // Build the M3U URL with the user's Bearer token embedded as a query param
        $token = $request->bearerToken();
        $baseUrl = config('app.url');

        return response()->json([
            'can_watch'    => true,
            'playlist_url' => "{$baseUrl}/api/user/playlist.m3u",
            'token'        => $token,
            'expires_at'   => $code->expires_at?->toDateTimeString(),
            'type'         => $code->type,
            'channel_count' => Channel::where('status', 'online')->count(),
        ]);
    }
}
